/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_RANKING_INTERFACE_H
#define _SCE_RANKING_INTERFACE_H

#include "np_toolkit_defines.h"

namespace sce {
	namespace Toolkit{
		namespace NP{

			///	@brief
			///	The namespace containing ranking functionality.
			///
			///	The namespace containing ranking functionality.	
			namespace Ranking {

			///	@brief
			///	Ranking interface class
			///
			///	The ranking interface class provides score registration and the retrieval of the current user's rank, a friend's rank and
			/// also a range of ranks. It allows a cache to be registered for the ranking service to use internally.
			class Interface{
			public:

				///	@brief
				///	Registers a ranking cache.
				///
				///	Registers a ranking cache. There are 4 types of ranking cache: a board cache where the board configuration is held,
				///	a write cache where scores awaiting server registration are held, a read cache where a range of scores are held,
				/// and a friend cache where scores of the user's friends are held for a single board. The table below contains details
				/// on the four types of cache:
				///
 				/// <table>
 				/// <tr><td>Board Cache</td>	<td>Each line is 24 bytes.</td></tr>
 				/// <tr><td>Write Cache</td>	<td>Each line is 289 bytes.</td></tr>
 				/// <tr><td>Read Cache</td>		<td>Each line is 416 bytes.</td></tr>
 				/// <tr><td>Friend Cache</td>	<td>40,024 bytes in size and capable of holding 101 users (100 friends and the current user). Contains <c>SceNpScoreRankData</c>,
				///	<c>SceNpScoreComment</c> and <c>SceNpScoreGameInfo</c> data as well as the board ID and the number of friends retrieved.</td></tr>
 				/// </table>
				///
				/// @param	boardLineCount	The number of lines to allocate for the board cache.
				///	@param	writeLineCount	The number of lines to allocate for the write cache. This  mainly for registering scores.
				///	@param	rangeLineCount	The number of lines to allocate for the read cache. This is used for range rank requests.
				///	@param	friendCache		A flag that specifies whether to create a friend cache.
				///
				/// @retval SCE_TOOLKIT_NP_SUCCESS						The operation was successful.							 
				///	@retval	SCE_TOOLKIT_NP_FAILED_ALLOCATE				The operation failed because memory could not be allocated for the cache.
				///	@retval	SCE_TOOLKIT_NP_OUT_OF_MEMORY				The operation failed because the %NP %Toolkit library could not allocate the required internal memory.
				static int registerCache(int boardLineCount, int writeLineCount, int rangeLineCount, bool friendCache);

				///	@brief
				///	Registers a user score.
				///
				///	Registers a user score. The sce::Toolkit::NP::RegisterScore data type should be used to pass in the score, comments and game
				/// information to be registered. There is server delay in processing the user's actual rank after a score has been registered,
				/// and a sce::Toolkit::NP::TempRank <c>Future</c> object can be used to receive a temporary rank back upon successful completion of the operation.
				/// If a ranking cache is being used, calling the registerCache() function will cause the ranking service to cache scores that there has been
				/// an attempt to register. Before pinging the server to register the score, the ranking service will check new scores against the cache to see if an
				/// attempt to register the new score has been made, or if the new score is lower
				/// than the one in the cache. This will save time and resources by avoiding trying to ping the server
				/// with lower scores or scores that there has already been an attempt to register.
				///
				/// @param	tempRank	A <c>Future</c> object with a sce::Toolkit::NP::TempRank datatype.
				///						Receives a temporary rank back.
				///	@param	score		The user's score to be registered.
				/// @param  async		A flag that indicates whether the function is non-blocking or blocking. 
				///						Defaults to true so the function is non-blocking by default.	
				///
				/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.							 
				///	@retval	SCE_TOOLKIT_NP_RANKING_SLOT_FULL		The operation failed because there were more than 32 simultaneous transactions going on at the time.
				///	@retval	SCE_TOOLKIT_NP_OUT_OF_MEMORY			The operation failed because the %NP %Toolkit library could not allocate the required internal memory.
				/// @retval SCE_TOOLKIT_NP_RANKING_NOT_HIGH_SCORE	The operation failed because the score was lower than user's current high score.
				///	@retval	rankingScoreRegistered				In the case of an asynchronous operation, this event will be passed to an event callback to signify success.
				///	@retval	rankingScoreRegisteredFail			In the case of an asynchronous operation, this event will be passed to an event callback to signify failure. For example, this could be due to server errors.
				///	@retval	rankingScoreRegisteredFailNotBest	In the case of an asynchronous operation, this event will be passed to an event callback if score registration failed because the score was not the best score.
				///	@retval	rankingServerError					In the case of an asynchronous operation, this event will be passed to an event callback if score registration failed because of a ranking server error (a timeout, for example).
				///
				/// @par Example:
				///	@code
				///	RankingService::RegScore score;
				///	memset(&score, 0x00, sizeof(score));
				///	score.boardId = 0;
				///	score.score = 9999999;
				///	std::strncpy(score.comment.data, "BLAHBLAHAAA", SCE_NP_SCORE_COMMENT_MAXLEN);
				///
				///	int ret = registerScore(score, true); // Asynchronous call
				///	@endcode
				static int registerScore(sce::Toolkit::NP::Utilities::Future<TempRank>* tempRank, const sce::Toolkit::NP::RegisterScore *score, bool async = true);

				///	@brief
				///	Retrieves a range of ranks for display purposes.
				///
				///	Retrieves a range of ranks to be displayed on the screen for the user. The maximum number of ranks that can be retrieved is
				///	30. It would be difficult for the user to see any more than this on the screen. When the operation has completed, the <c>Future</c> object
				/// passed to <c><i>score</i></c> needs to be polled to see if the buffer is filled or not. If registerCache()has been called to register a ranking cache,
				/// the ranks will be held in the cache. This means that the next time the same scores are required they will be retrieved
				///	from a cache instead of pinging the server. This saves time and prevents server overload.
				///
				///	@param	score		Output. A <c>Future</c> object with a sce:Toolkit::NP::RankInformation data type, which receives the list of ranks to be retrieved.
				///	@param	boardId		The board ID from which the ranks that needs to be retrieved.
				///	@param	startRank	The starting rank to be retrieved.
				///	@param	range		The number of ranks, starting from <c><i>startRank</i></c>, to be retrieved.
				/// @param  async		A flag that indicates whether the function is non-blocking or blocking. 
				///						Defaults to false so the function is blocking by default.	
				///
				/// @retval SCE_TOOLKIT_NP_SUCCESS				The operation was successful.							 
				///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because an invalid pointer was passed to <c><i>score</i></c>.
				///	@retval	SCE_TOOLKIT_NP_OUT_OF_MEMORY		The operation failed because the %NP %Toolkit library could not allocate the required internal memory.
				///	@retval	SCE_TOOLKIT_NP_RANKING_SLOT_FULL	The operation failed because there were more than 32 simultaneous transactions going on at the time.
				///	@retval SCE_TOOLKIT_NP_RANKING_NO_SCORES	The operation failed because there were no scores.
				///	@retval rankingRangeRetrieved			In the case of an asynchronous operation, this event will be passed to an event callback to signify success.
				///	@retval	rankingRangeRetrievedFail		In the case of an asynchronous operation, this event will be passed to an event callback if the retrieval failed because the <c><i>range</i></c> or <c><i>boardId</i></c> arguments were invalid.
				///	@retval	rankingServerError				In the case of an asynchronous operation, this event will be passed to an event callback if the retrieval failed because of a ranking server error (a timeout, for example).
				///
				/// @note	If the function is called synchronously, it returns the number of ranks retrieved upon normal termination. 
				static int displayRangeOfRanks(	sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::RankInformation> * score,
												SceNpScoreBoardId boardId,
												int startRank,
												int range,
												bool async = true);

				///	@brief
				///	Retrieves the rank of a friend of the user for display purposes.
				///
				/// Retrieves the rank of a friend of the user for display purposes. When the operation has completed, the <c>Future</c> object
				/// passed to <c><i>friendScore</i></c> needs to be polled to see if the buffer is filled or not. If registerCache()
				/// has been called to register a ranking cache, the friend's rank will be held in the cache. This means the next time the score is required
				/// it will be retrieved from the cache instead of pinging the server again. This saves time and prevents server overload.
				///
				/// @param	friendScore	A <c>Future</c> object with a sce::Toolkit::NP::TempRank data type.
				///						Receives the friend's temporary rank back.
				///	@param	boardId		The ID of the board that the friend's rank is to be retrieved from.
				/// @param  async		A flag that indicates whether the function is non-blocking or blocking. 
				///						Defaults to false so the function is blocking by default.
				///
				/// @retval SCE_TOOLKIT_NP_SUCCESS						The operation was successful.							 
				///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER				The operation failed because an invalid pointer was passed to <c><i>score</i></c>.
				///	@retval	SCE_TOOLKIT_NP_OUT_OF_MEMORY				The operation failed because the %NP %Toolkit library could not allocate the required internal memory.
				///	@retval	SCE_TOOLKIT_NP_RANKING_SLOT_FULL			The operation failed because there were more than 32 simultaneous transactions going on at the time.
				///	@retval SCE_TOOLKIT_NP_RANKING_NO_SCORES			The operation failed because there were no scores.
				///	@retval	rankingFriendsRetrieved					In the case of an asynchronous operation, this event will be passed to an event callback to signify success.
				///	@retval rankingFriendsRetrievedFail				In the case of an asynchronous operation, this event will be passed to an event callback if the retrieval failed (general error).
				///	@retval	rankingFriendsRetrievedFailNoFriends	In the case of an asynchronous operation, this event will be passed to an event callback if the retrieval failed because the user has no friends.	
				///	@retval	rankingServerError						In the case of an asynchronous operation, this event will be passed to an event callback if the retrieval failed because of a ranking server error (a timeout, for example).
				static int displayFriendRank(	sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::FriendsRankInformation>* friendScore,
												SceNpScoreBoardId boardId,
												bool async = true);

				///	@brief
				///	Retrieves the user's own rank.
				///
				///	Retrieves the user's own rank.
				///	This function retrieves the ranking information of the user whose <c>SceNPId</c> is passed in. You can retrieve current user's rank information by passing <c>NULL</c>
				/// to <c><i>npId</i></c>.
				///
				///	@param	currentUserScore	Output. A <c>Future</c> object with a sce:Toolkit::NP::RankInformation data type, which receives the ranks of the %NP ID passed in.
				///	@param	npId				The <c>SceNPId</c> of the user to be retrieved.
				///	@param	boardId				The board ID from which the ranks need to be retrieved.
				/// @param  async				A flag that indicates whether the function is non-blocking or blocking.
				///								Defaults to false so the function is blocking by default.	
				///
				/// @retval SCE_TOOLKIT_NP_SUCCESS						The operation was successful.							 
				///	@retval	SCE_TOOLKIT_NP_INVALID_POINTER				The operation failed because an invalid pointer was passed to <c><i>score</i></c>.
				///	@retval	SCE_TOOLKIT_NP_OUT_OF_MEMORY				The operation failed because the %NP %Toolkit library could not allocate the required internal memory.
				///	@retval	SCE_TOOLKIT_NP_RANKING_SLOT_FULL			The operation failed because there were more than 32 simultaneous transactions going on at the time.
				///	@retval	rankingUserRankRetrieved					In the case of an asynchronous operation, this event will be passed to an event callback to signal that user rank has been retrieved successfully.
				///	@retval	rankingUserRankRetrievedFailed				In the case of an asynchronous operation, this event will be passed to an event callback to signal that user rank has been retrieved unsuccessfully.
				static int displayUserRank(	sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::UserRankInformation>* currentUserScore,
											SceNpId npId, SceNpScoreBoardId boardId,
											bool async = true);
		
				///	@brief
				///	Terminates the ranking service.
				///
				///	Terminates the ranking service. This function free up ranking cache memory registered by the registerCache(),
				///	and calls <c>sceNpScoreDestroyTitleCtx()</c> and <c>sceNpScoreTerm()</c> internally. The <c>SCE_SYSMODULE_NP_SCORE_RANKING</c>
				/// module will be unloaded as well. For error messages, please refer to the descriptions for <c>sceNpScoreDestroyTitleCtx()</c> and <c>sceNpScoreTerm()</c>
				/// in the <em>NP_ScoreRanking-Reference</em> document. Note that this is a synchronous function.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS						The ranking service was successfully terminated.
				static int rankingTerm();
		
				///	@brief
				///	Initializes the ranking service.
				///
				///	Initializes the ranking service. There no need to call this function when the %NP %Toolkit library is first time initialized
				/// and the ranking service is automatically brought up. It only needs to be called when the ranking service has been manually terminated 
				///	by calling rankingTerm(). This function calls <c>sceNpScoreInit()</c> and <c>sceNpScoreCreateTitleCtx()</c> internally and loads the <c>SCE_SYSMODULE_NP_SCORE_RANKING</c>
				/// module as well. For error messages, please refer to the descriptions for <c>sceNpScoreInit()</c> and <c>sceNpScoreCreateTitleCtx()</c>
				/// in the <em>NP_ScoreRanking-Reference</em> document. Note that this is a synchronous function.
				///
				///	@retval	SCE_TOOLKIT_NP_SUCCESS					The ranking service was successfully initialized.
				/// @retval SCE_TOOLKIT_NP_RANKING_NOT_TERMINATED	The ranking service has not been terminated since the last time it was initialized.
				static int rankingInit();

			};

			}//end Ranking
		}//end NP
	}//end Toolkit
}//end sce
#endif
